package com.infoworld.widgetservice.web;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Optional;

import com.infoworld.widgetservice.model.Widget;
import com.infoworld.widgetservice.service.WidgetService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class WidgetController {
    @Autowired
    private WidgetService widgetService;

    @GetMapping("/widget/{id}")
    public ResponseEntity<?> getWidget(@PathVariable Long id) {
        return widgetService.findById(id)
                .map(widget -> {
                    try {
                        return ResponseEntity
                                .ok()
                                .location(new URI("/widget/" + id))
                                .eTag(Integer.toString(widget.getVersion()))
                                .body(widget);
                    } catch (URISyntaxException e) {
                        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
                    }
                })
                .orElse(ResponseEntity.notFound().build());
    }

    @GetMapping("/widgets")
    public List<Widget> getWidgets() {
        return widgetService.findAll();
    }

    @PostMapping("/widgets")
    public ResponseEntity<?> createWidget(@RequestBody Widget widget) {
        Widget newWidget = widgetService.create(widget);
        try {
           return ResponseEntity
                   .created(new URI("/widget/" + newWidget.getId()))
                   .eTag(Integer.toString(newWidget.getVersion()))
                   .body(newWidget);
        } catch (URISyntaxException e) {
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PutMapping("/widget/{id}")
    public ResponseEntity<?> updateWidget(@PathVariable Long id, @RequestBody Widget widget, @RequestHeader("If-Match") Integer ifMatch) {
        Optional<Widget> existingWidget = widgetService.findById(id);
        return existingWidget.map(w -> {
            if (w.getVersion() != ifMatch) {
                return ResponseEntity.status(HttpStatus.CONFLICT).build();
            }

            w.setName(widget.getName());
            w.setVersion(w.getVersion() + 1);

            Widget updatedWidget = widgetService.save(w);
            try {
                return ResponseEntity.ok()
                        .location(new URI("/widget/" + updatedWidget.getId()))
                        .eTag(Integer.toString(updatedWidget.getVersion()))
                        .body(updatedWidget);
            } catch (URISyntaxException e) {
                throw new RuntimeException(e);
            }
        }).orElse(ResponseEntity.notFound().build());
    }

    @DeleteMapping("widget/{id}")
    public ResponseEntity<?> deleteWidget(@PathVariable Long id) {
        Optional<Widget> existingWidget = widgetService.findById(id);
        return existingWidget.map(w -> {
           widgetService.deleteById(w.getId());
           return ResponseEntity.ok().build();
        }).orElse(ResponseEntity.notFound().build());
    }
}
