package com.infoworld.widgetservice.repository;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.infoworld.widgetservice.model.Widget;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.boot.test.autoconfigure.orm.jpa.TestEntityManager;

@DataJpaTest
public class WidgetRepositoryTest {
    @Autowired
    private TestEntityManager entityManager;

    @Autowired
    private WidgetRepository widgetRepository;

    private final List<Long> widgetIds = new ArrayList<>();

    private final List<Widget> testWidgets = Arrays.asList(
            new Widget("Widget 1", 1),
            new Widget("Widget 2", 1),
            new Widget("Widget 3", 1)
    );

    @BeforeEach
    void setup() {
        testWidgets.forEach(widget -> {
            entityManager.persist(widget);
            widgetIds.add((Long)entityManager.getId(widget));
        });
        entityManager.flush();
    }

    @AfterEach
    void tearDown() {
        widgetIds.forEach(id -> {
            Widget widget = entityManager.find(Widget.class, id);
            if (widget != null) {
                entityManager.remove(widget);
            }
        });
        widgetIds.clear();
    }

    @Test
    void testFindAll() {
        List<Widget> widgetList = widgetRepository.findAll();
        assertEquals(3, widgetList.size());
    }

    @Test
    void testFindById() {
        Widget widget = widgetRepository.findById(widgetIds.getFirst()).orElse(null);

        assertNotNull(widget);

        assertEquals(widgetIds.getFirst(), widget.getId());
        assertEquals("Widget 1", widget.getName());
        assertEquals(1, widget.getVersion());
    }

    @Test
    void testFindByIdNotFound() {
        Widget widget = widgetRepository.findById(widgetIds.getFirst() + testWidgets.size()).orElse(null);
        assertNull(widget);
    }

    @Test
    void testCreateWidget() {
        Widget widget = new Widget("New Widget", 1);
        Widget insertedWidget = widgetRepository.save(widget);

        assertNotNull(insertedWidget);
        assertEquals("New Widget", insertedWidget.getName());
        assertEquals(1, insertedWidget.getVersion());

        widgetIds.add(insertedWidget.getId());
    }

    @Test
    void testFindByName() {
        List<Widget> found = widgetRepository.findByName("Widget 2");
        assertEquals(1, found.size(), "Expected to find 1 Widget");

        Widget widget = found.getFirst();
        assertEquals("Widget 2", widget.getName());
        assertEquals(1, widget.getVersion());
    }
}
