package com.infoworld.widgetservice.web;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.header;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.infoworld.widgetservice.model.Widget;
import com.infoworld.widgetservice.service.WidgetService;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.WebMvcTest;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.test.context.bean.override.mockito.MockitoBean;
import org.springframework.test.web.servlet.MockMvc;

@WebMvcTest(WidgetController.class)
public class WidgetControllerTest {
    @Autowired
    private MockMvc mockMvc;

    @MockitoBean
    private WidgetService widgetService;

    @Test
    void testGetWidgetById() throws Exception {
        Widget widget = new Widget(1L, "My Widget", 1);
        when(widgetService.findById(1L)).thenReturn(Optional.of(widget));

        mockMvc.perform(get("/widget/{id}", 1))
                // Validate that we get a 200 OK Response Code
                .andExpect(status().isOk())

                // Validate Headers
                .andExpect(content().contentType(MediaType.APPLICATION_JSON))
                .andExpect(header().string(HttpHeaders.LOCATION, "/widget/1"))
                .andExpect(header().string(HttpHeaders.ETAG, "\"1\""))

                // Validate content
                .andExpect(jsonPath("$.id").value(1L))
                .andExpect(jsonPath("$.name").value("My Widget"))
                .andExpect(jsonPath("$.version").value(1));
    }

    @Test
    void testGetWidgetByIdNotFound() throws Exception {
        when(widgetService.findById(1L)).thenReturn(Optional.empty());

        mockMvc.perform(get("/widget/{id}", 1))
                // Validate that we get a 404 Not Found Response Code
                .andExpect(status().isNotFound());
    }

    @Test
    void testGetWidgets() throws Exception {
        List<Widget> widgets = new ArrayList<>();
        widgets.add(new Widget(1L, "Widget 1", 1));
        widgets.add(new Widget(2L, "Widget 2", 1));
        widgets.add(new Widget(3L, "Widget 3", 1));

        when(widgetService.findAll()).thenReturn(widgets);

        mockMvc.perform(get("/widgets"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.length()").value(3))
                .andExpect(jsonPath("$[0].id").value(1L))
                .andExpect(jsonPath("$[0].name").value("Widget 1"))
                .andExpect(jsonPath("$[0].version").value(1));
    }

    @Test
    void testCreateWidget() throws Exception {
        Widget widget = new Widget(1L, "Widget 1", 1);
        when(widgetService.create(any())).thenReturn(widget);

        mockMvc.perform(post("/widgets")
                .contentType(MediaType.APPLICATION_JSON)
                .content("{\"name\": \"Widget 1\"}"))

                // Validate that we get a 201 Created Response Code
                .andExpect(status().isCreated())

                // Validate Headers
                .andExpect(content().contentType(MediaType.APPLICATION_JSON))
                .andExpect(header().string(HttpHeaders.LOCATION, "/widget/1"))
                .andExpect(header().string(HttpHeaders.ETAG, "\"1\""))

                // Validate content
                .andExpect(jsonPath("$.id").value(1L))
                .andExpect(jsonPath("$.name").value("Widget 1"))
                .andExpect(jsonPath("$.version").value(1));
    }

    @Test
    public void testSuccessfulUpdate() throws Exception {
        // Create a mock Widget when the WidgetService's findById(1L) is called
        Widget mockWidget = new Widget(1L, "Widget 1", 5);
        when(widgetService.findById(1L)).thenReturn(Optional.of(mockWidget));

        // Create a mock Coffee that is returned when the CoffeeController saves the Coffee to the database
        Widget savedWidget = new Widget(1L, "Updated Widget 1", 6);
        when(widgetService.save(any())).thenReturn(savedWidget);

        // Execute a PUT /widget/1 with a matching version: 5
        mockMvc.perform(put("/widget/{id}", 1L)
                        .contentType(MediaType.APPLICATION_JSON)
                        .header(HttpHeaders.IF_MATCH, 5)
                        .content("{\"id\": 1, \"name\":  \"Updated Widget 1\"}"))

                // Validate that we get a 200 OK HTTP Response
                .andExpect(status().isOk())

                // Validate the headers
                .andExpect(content().contentType(MediaType.APPLICATION_JSON))
                .andExpect(header().string(HttpHeaders.LOCATION, "/widget/1"))
                .andExpect(header().string(HttpHeaders.ETAG, "\"6\""))

                // Validate the contents of the response
                .andExpect(jsonPath("$.id").value(1L))
                .andExpect(jsonPath("$.name").value("Updated Widget 1"))
                .andExpect(jsonPath("$.version").value(6));
    }

    @Test
    public void testUpdateConflict() throws Exception {
        // Create a mock coffee with a version set to 5
        Widget mockWidget = new Widget(1L, "Widget 1", 5);

        // Return the mock Coffee when the CoffeeService's findById(1L) is called
        when(widgetService.findById(1L)).thenReturn(Optional.of(mockWidget));

        // Execute a PUT /widget/1 with a mismatched version number: 2
        mockMvc.perform(put("/widget/{id}", 1L)
                        .contentType(MediaType.APPLICATION_JSON)
                        .header(HttpHeaders.IF_MATCH, 2)
                        .content("{\"id\": 1, \"name\":  \"Updated Widget 1\"}"))

                // Validate that we get a 409 Conflict HTTP Response
                .andExpect(status().isConflict());
    }

    @Test
    public void testUpdateNotFound() throws Exception {
        // Return the mock Coffee when the CoffeeService's findById(1L) is called
        when(widgetService.findById(1L)).thenReturn(Optional.empty());

        // Execute a PUT /coffee/1 with a mismatched version number: 2
        mockMvc.perform(put("/widget/{id}", 1L)
                        .contentType(MediaType.APPLICATION_JSON)
                        .header(HttpHeaders.IF_MATCH, 2)
                        .content("{\"id\": 1, \"name\":  \"Updated Coffee 1\"}"))

                // Validate that we get a 404 Not Found HTTP Response Code
                .andExpect(status().isNotFound());
    }

    @Test
    void testDeleteSuccess() throws Exception {
        // Setup mocked product
        Widget mockWidget = new Widget(1L, "Widget 1", 5);

        // Setup the mocked service
        when(widgetService.findById(1L)).thenReturn(Optional.of(mockWidget));
        doNothing().when(widgetService).deleteById(1L);

        // Execute our DELETE request
        mockMvc.perform(delete("/widget/{id}", 1L))
                .andExpect(status().isOk());
    }

    @Test
    void testDeleteNotFound() throws Exception {
        // Setup the mocked service
        when(widgetService.findById(1L)).thenReturn(Optional.empty());

        // Execute our DELETE request
        mockMvc.perform(delete("/widget/{id}", 1L))
                .andExpect(status().isNotFound());
    }

}
